#include <concepts>
#include <type_traits>
#include <memory>
#include <vector>
#include <iostream>
#include <optional>
#include <cxxabi.h>

using namespace std::string_literals;

struct Object {
	int value = 0;

	Object(int valueIn) : value{valueIn} { }

	friend std::ostream& operator<<(std::ostream& to, const Object& o) {
		return to << "Object{" << o.value << "}";
	}
};

std::string show(int v)
{
	return "INT "s + std::to_string(v);
}

template <typename T>
concept smart_pointer = requires(T ptr)
{
	{ *ptr };
	{ ptr.get() };
	{ ptr.operator->() };
};

template <typename T>
concept any_pointer = std::is_pointer_v<T> || smart_pointer<T>;

template <typename T>
concept optional_type = requires(T ptr)
{
	{ *ptr };
	{ ptr.has_value() } -> std::same_as<bool>;
	{ ptr.value() };
};

template <typename T>
concept dereferenceable = requires(T ptr)
{
	{ *ptr };
	{ static_cast<bool>(ptr) };
};

#if defined(SPECIFIC)
#define SMART_POINTER
#define OPTIONAL
#endif

template <typename T>
std::string show(const std::string& label, const T& ptr)
{
	if (!ptr) {
		return label + "<NULL>";
	}
	return label + ":" + show(*ptr);
}

#if defined(RAW_POINTER)
template <typename T> requires std::is_pointer_v<T>
std::string func(const T& ptr)
{
	return show("RawPTR", ptr);
}
#endif

#if defined(SMART_POINTER)
template <smart_pointer T>
std::string func(const T& ptr)
{
	return show("SmartPTR", ptr.get());
}
#endif

#if defined(OPTIONAL)
template <optional_type T>
std::string func(const T& opt)
{
	return show("OPT", (opt.has_value() ? &opt.value() : nullptr));
}
#endif

#if defined(ANY_POINTER)
template <any_pointer T>
std::string func(const T& ptr)
{
	return show("PTR", ptr);
}
#endif

#if defined(DEREF)
template <typename T> requires (dereferenceable<T>
#if defined(RAW_POINTER)
	       	&& !std::is_pointer_v<T>
#endif
#if defined(SMART_POINTER)
	       	&& !smart_pointer<T>
#endif
#if defined(OPTIONAL)
	       	&& !optional_type<T>
#endif
#if defined(ANY_POINTER)
	       	&& !any_pointer<T>
#endif
		)
std::string func(const T& opt)
{
	return show("DEREF", opt);
}
#endif

template <typename... T>
std::string func(T&&...)
{
	return "<UNMATCHED>";
}

template <typename T>
void do_f(const std::string& label, const T& o)
{
	const auto& ti = typeid(T);
	int status;
	const char * name = abi::__cxa_demangle(ti.name(), NULL, NULL, &status);
	std::cout << label << ": " << name << " (" << status << "): " << func(o) << std::endl;
}

#define f(expr) do_f(#expr, (expr))

int main()
{
	f(nullptr);

	int intValues[] = { 1, 2, 3, 4 };
	f(intValues);
	f(intValues[0]);
	f(&intValues[0]);

	std::vector<int> intVector = { 2, 2, 2, 2 };
	f(intVector);
	f(intVector.data());
	f(intVector[0]);
	f(&intVector[0]);

	int anInt = 42;
	f(anInt);
	f(&anInt);

	auto* anIntPtr = &anInt;
	f(anIntPtr);

	int* aNullPtr = nullptr;
	f(aNullPtr);

	auto aUniquePtr = std::make_unique<int>(666);
	f(aUniquePtr);

	auto aSharedPtr = std::make_shared<int>(999);
	f(aSharedPtr);

	auto anOptional = std::make_optional<int>(1234);
	f(anOptional);
}

